/*
 *	Phoneme test application
 *
 * (c) 2025 Allard van der Bas / Digital Fluff
 */
#include <coleco.h>
#include <string.h>

// Our data files.
// #include "Alphabet16x16.h"
#include "Alphabet24x24.h"
#include "square_24x24.h"
#include "font8x8.h"

// SN76489 manipulation assembly routines
#include "SN76489A.h"

// Our phonemes
#include "phonemes.h"
// Some words to say
#include "ARPAbet_words.h"

// Insults
#include "adjectives.h"
#include "nouns.h"

// Visuals for 'SAY' application
#include "ARPAbet_kb_176_80.h"
#include "square16x16.h"
#include "char_sprite.h"

/*
 * nmi / color cycling.
 */
volatile unsigned char color_offset;

static const u8 colors[] = {
    0xd,0xd,0xd,0x5,
    0xd,0x5,0x5,0x5,
    0xc,0x5,0xc,0xc,
    0xc,0x2,0xc,0x2,
    
    0x2,0x2,0x3,0x2,
    0x3,0x3,0x3,0xa,
    0x3,0xa,0xa,0xa,
    0x9,0xa,0x9,0x9,
    0x9,0x8,0x9,0x8,
    
    0x8,0x8,0xd,0x8,
    0xd,0xd,0xd,0x5,
    0xd,0x5,0x5,0x5,
    0xc,0x5,0xc,0xc,
    0xc,0x2,0xc,0x2
};

volatile unsigned char ispal;

void nmi(void) {
	// update sprite display during nmi, if you enable sprites
	if (spr_enable) {
		color_offset++;
		
	    if (color_offset > 39) {
			color_offset = 0;
		}
		
	    sprites[0].colour = colors[color_offset];
	    sprites[1].colour = colors[color_offset];
	    
		spr_update();
	}
	
	// Get random number spinning.	
	sys_random();
}

//=-=-=-=-=-

/*
 * Draw a char with color in VRAM (mode2bmp).
 */
void draw8x8at(unsigned char whereX,unsigned char whereY,unsigned char *what,unsigned char color) {
	vdp_blocknmi();
	vdp_putvram(0x0000+whereY*32*8 + whereX*8,what,8);
	vdp_fillvram(0x2000+whereY*32*8 + whereX*8,color,8);
	vdp_releasenmi();
}

/*
 * Draw a char with current colors.
 */
void draw8x8at_nocolor(unsigned char whereX,unsigned char whereY,unsigned char *what) {
	vdp_blocknmi();
	vdp_putvram(0x0000+whereY*32*8 + whereX*8,what,8);
	vdp_releasenmi();
}


/*
 * Change the color of a number of chars in VRAM (mode2bmp).
 */
void color8x8at(unsigned char whereX,unsigned char whereY,unsigned char howmany,unsigned char color) {
	vdp_blocknmi();
	vdp_fillvram(0x2000+whereY*32*8 + whereX*8,color,8*howmany);
	vdp_releasenmi();
}

/*
 * Mode 0 txt.
 */
void drawcharpixel(unsigned char x,unsigned char y,unsigned char which) {
	vdp_blocknmi();
	vdp_fillvram(0x1800+y*32+x,which,1);
	vdp_releasenmi();
}

void draw_string(unsigned char x,unsigned char y,unsigned char *what,unsigned char length) {
	unsigned char cnt;
	
	for(cnt=0;cnt<length;cnt++) {
		drawcharpixel(x+cnt,y,what[cnt] - ' ');
	}
}


/*
 * Draw a 24x24 character at text position
 *      - whereX 0..31  (8 bit boundary X).
 *      - whereY 0..23  (8 bit boundary Y).
 *      - which  0..25  (character A..Z) | 0xFE select square | 0xFF normal_square.
 *      - color  <4 bit>|<4 bit> (color).
 *      - select 0|1 (use the select square or the not select square).
 */
void draw24x24at(unsigned char whereX,unsigned char whereY,unsigned char which,unsigned char color,unsigned char select) {
	unsigned char what[72];
	int cnt;
	
	if ( (which != 0xFF) && (which != 0xFE) ) {

		if (which <= 25) {
			memcpy(what,&Alphabet24x24[which*9*8],72);
		}	
		
		// Select or square it.
		for(cnt=0;cnt<72;cnt++) {
		  if (select == 0) {
			what[cnt] &= square_24x24[cnt];
		  } else {
			// what[cnt] &= square_24x24_select[cnt];
		  }
		}

	} else if (which == 0xFE) {
		// memcpy(what,square_24x24_select,72);
	} else {
		memcpy(what,square_24x24,72);
	}
		
	draw8x8at(whereX,whereY,what,color);
	draw8x8at(whereX+1,whereY,what+8,color);
	draw8x8at(whereX+2,whereY,what+16,color);
	
	draw8x8at(whereX,whereY+1,what+24,color);
	draw8x8at(whereX+1,whereY+1,what+32,color);
	draw8x8at(whereX+2,whereY+1,what+40,color);
	
	draw8x8at(whereX,whereY+2,what+48,color);
	draw8x8at(whereX+1,whereY+2,what+56,color);
	draw8x8at(whereX+2,whereY+2,what+64,color);
}

/*
 * Simple string print.
 */
void printAt(unsigned char x,unsigned char y,char *what,unsigned char color) {
	int len;
	unsigned char cnt;
	unsigned char toprint;
	
	len=strlen(what);
	for(cnt=0;cnt<len;cnt++) {
		toprint = (what[cnt] - ' ');		
		draw8x8at(x+cnt,y,&font8x8[ toprint*8 ],color);
	}
}

void clear_line(unsigned char y) {
	vdp_blocknmi();
	vdp_fillvram(0x0000 + y*32*8,0,32*8);
	vdp_releasenmi();	
}

void printCentered(unsigned char y,char *what) {
	int len;
	unsigned char cnt;
	unsigned char toprint;
	unsigned char x;
	
	clear_line(y);
	
	len = strlen(what);
	
	x = (32-len)/2;
	
	for(cnt=0;cnt<len;cnt++) {
		toprint = (what[cnt] - ' ');		
		draw8x8at(x+cnt,y,&font8x8[ toprint*8 ],0xF1);
	}

}

void setup_screen(void) {
	unsigned short cnt;
	
	// Put screen in bmp mode 2
	vdp_setmode2bmp();
	
	vdp_blocknmi();
	vdp_fillvram(0x1000,0,0x800);
	vdp_fillvram(0x0800,0,0x800);
	vdp_fillvram(0x0000,0,0x800);
	
	// Put values 0..256 in 3 different memory regions.
	for(cnt=0;cnt<768;cnt++) {
		vdp_fillvram(0x1800+cnt,cnt&0xFF,1);
	}

	// Set screen color to White on black for whole screen.
	vdp_fillvram(0x2000,0xF1,32*24*8);
	
	// Turn off sprites as well.
	vdp_fillvram(0x3800,0,0x800);
	
	vdp_releasenmi();
}

/*
 * Show an intro screen.
 *
 * DIGITAL FLUFF
 *
 *   PRESENTS
 *
 */
void show_intro(void) {
	unsigned char *palntsc;
	
	// Setup default screen.
	setup_screen();

	draw24x24at(5,5,'d'-'a',0x1D,0);
	draw24x24at(8,5,'i'-'a',0x18,0);
	draw24x24at(11,5,'g'-'a',0x19,0);
	draw24x24at(14,5,'i'-'a',0x1B,0);
	draw24x24at(17,5,'t'-'a',0x13,0);
	draw24x24at(20,5,'a'-'a',0x12,0);
	draw24x24at(23,5,'l'-'a',0x14,0);	

	draw24x24at(8,8,'f'-'a',0xF1,0);
	draw24x24at(11,8,'l'-'a',0xF1,0);
	draw24x24at(14,8,'u'-'a',0xF1,0);
	draw24x24at(17,8,'f'-'a',0xF1,0);
	draw24x24at(20,8,'f'-'a',0xF1,0);
	
	draw24x24at(4,15,'p'-'a',0x1F,0);
	draw24x24at(7,15,'r'-'a',0x1F,0);
	draw24x24at(10,15,'e'-'a',0x1F,0);
	draw24x24at(13,15,'s'-'a',0x1F,0);
	draw24x24at(16,15,'e'-'a',0x1F,0);
	draw24x24at(19,15,'n'-'a',0x1F,0);
	draw24x24at(22,15,'t'-'a',0x1F,0);
	draw24x24at(25,15,'s'-'a',0x1F,0);
	
	palntsc = (unsigned char *)0x0069;
	
	printAt(5,21,"12 KB 2BIT DPCM VERSION",0xF1);
	printAt(1,23,"V1.3",0xF1);
	
	if (*palntsc == 60) {
		printAt(28,23,"NTSC",0xF1);
		
		ispal = 0;
	} else {
		printAt(28,23,"PAL",0xF1);
		
		ispal = 1;
	}
	
	vdp_enablescr();
	vdp_enablenmi();

	vdp_waitvblank(150);
}


/*
 * SN routines.
 */ 
volatile unsigned char speed;

void sound_write_register(unsigned char value) {
	write_sn76489A_register(value);
}

void ready_soundchip_forsample(void) {
	unsigned char value;
	
	// Make sure sound is not muted in the nmi.
	// And possibly overwriting SN values.
	snd_mute = 1;
	
	// Setup soundchip.
    
	// Zero the volumes.
    value = 0xFF;	// %1 111 1111
    sound_write_register(value);
    
    value = 0xDF;	// %1 101 1111
    sound_write_register(value);
    
    value = 0xBF;	// %1 011 1111
    sound_write_register(value);
    
    value = 0x9F;   // %1 001 1111
    sound_write_register(value);
    
    // Period to 1 on tone channels (1, 2, 3)
    value = 0xC1;	// %1 100 0001 
    sound_write_register(value);
    value = 0x00;	// %0 x 000000
    sound_write_register(value);
    value = 0xA1;	// %1 010 0001
    sound_write_register(value);
    value = 0x00;	// %0 x 000000
    sound_write_register(value);
    value = 0x81;	// %1 100 0001
    sound_write_register(value);
    value = 0x00;	// %0 x 000000
    sound_write_register(value);
}

void play_sample(unsigned char *from,unsigned int length) {		
	vdp_blocknmi();
	
	// Setup soundchip.
    ready_soundchip_forsample();
    
	sound_play_sample_dpcm(speed,from,length);
		
	vdp_releasenmi();
	
	// Set nmi to mute the sound again.
	snd_mute = 0;
}

/*
 * if inlength == 0, take length from first byte
 * else, use lengt
 */
void speak(unsigned char *which,unsigned char inlength) {
	unsigned char cnt;
	unsigned char length;
	
	vdp_blocknmi();
	
	// Setup soundchip.
    ready_soundchip_forsample();

	if (inlength == 0) {
		length = which[0];
		length++;
		cnt = 1;
	} else {
		length = inlength;
		cnt = 0;
	}

	for(cnt;cnt<length;cnt++) {
		
		if (which[cnt] == PSE) {
			sound_sample_delay();
		} else if (which[cnt] == AA) {
			sound_play_sample_dpcm(speed,&ph_AA_2bitDPCM[phonemes[0]],phoneme_lengths[0]);
		} else {
			sound_play_sample_dpcm(speed,&ph_AA_2bitDPCM[phonemes[which[cnt]]],phoneme_lengths[which[cnt]]);
		}
		
	}

	vdp_releasenmi();
	
	// Set nmi to mute the sound again.
	snd_mute = 0;
	
}

void insult_me(void) {
	unsigned char done;
	unsigned char *phonemes;
	unsigned char *word;
	unsigned char rndnum;
	
	setup_screen();
	printAt(8,1,"PRESS AND HOLD *",0xF1);
	printAt(4,2,"FOR THE HURTING TO STOP",0xF1);
	vdp_enablescr();
	vdp_enablenmi();

	done = 0;
		
	while(!done) {
		rndnum = sys_random() % 100;
		
		phonemes = (unsigned char *)phon_adjective[rndnum];
		word = (unsigned char *)adjective[rndnum];

		// print the word
		printCentered(10,word);
		speak(phonemes,strlen(phonemes));

		vdp_waitvblank(30);
		
		rndnum = sys_random() % 100;
				
		phonemes = (unsigned char *)phon_nouns[rndnum];
		word = (unsigned char *)noun[rndnum];			

		// print the word
		printCentered(10,word);
		speak(phonemes,strlen(phonemes));
			
		vdp_waitvblank(30);
		
		if ( (keypad_1 == PAD_KEYSTAR) ) {
			done = 1;
			
			while(keypad_1 == PAD_KEYSTAR) {
				vdp_waitvblank(1);
			}
			
		}		
	}
}

const unsigned char kbphoneme1[][3] = {
	"AA","AE","AH","AO","AW","AY","B ","CH","D ","DH","  ",
	"EH","ER","EY","F ","G ","HH","IH","IY","JH","K ",
	"L ","M ","N ","NG","OW","OY","P ","R ","S ","  ",
	"SH","T ","TH","UH","UW","V ","W ","Y ","Z ","ZH",
	"  "
};

void show_keyboard(void) {
	unsigned char cntx;
	unsigned char cnty;
	
	for(cntx=0;cntx<22;cntx++) {
		for(cnty=0;cnty<10;cnty++) {
			draw8x8at_nocolor(5+cntx,14+cnty,&ARPAbetkb[cnty*22*8 + cntx*8]);
		}
	}

}

void setup_initialkb_charandsprite(void) {
	show_keyboard();

	vdp_blocknmi();

	// Put in the 8x8 font in 2 regions.
	vdp_putvram(0x0000,font8x8,sizeof(font8x8));
	vdp_putvram(0x0800,font8x8,sizeof(font8x8));
	
	// Now send the sprite to sprite RAM.
	vdp_putvram(0x3800,square16x16,32);
  	vdp_putvram(0x3800+32,char_select,32);
  
	vdp_releasenmi();
	
	// Enable screen.
	vdp_enablescr();
		
	// Clear all the sprites.
	spr_clear();
		
	// setup sprites.
  	sprites[0].pattern = 0;
  	sprites[1].pattern = 4;
    
  	sprites[0].x = 5*8;
  	sprites[0].y = (14*8) - 1;
  
  	sprites[1].x = 4*8;
  	sprites[1].y = 9*8;
}

// The text to display.
unsigned char input_pointer;
unsigned char input[128];

// The phonemes to speak.
unsigned char input_phoneme_pointer;
unsigned char input_phoneme[128];

void clear_input(void) {
	unsigned char row;
	
	// Clear input.
	for(row=0;row<128;row++) {
		input[row] = ' ';
	}
}

void handle_key_phonemes(unsigned char row,unsigned char col) {
	unsigned char key;
	unsigned char key_value;
	unsigned char cnt;
	
	// Find the key being pressed.
	key = col;
	
	if (row == 0) {
		// Do nothing.
	} else if (row == 1) {
		key += 11;
	} else if (row == 2) {
		key += 21;
	} else if (row == 3) {
		key += 31;
	} else if (row == 4) {
		key += 41;
	}
	
	// DEL
	if (key == 10) {
		
		if (input_pointer > 4) {
			input_pointer--;
			input_pointer--;
			
			// Back 1
			input_phoneme_pointer--;
			
			if (sprites[1].x == 0) {
				sprites[1].y -= 8;
			}
						
			sprites[1].x -= 16;
		}
		
		input[input_pointer] = ' ';
		input[input_pointer+1] = ' ';
	
	// ENTER
	} else if (key == 30) {
		
		// If there is input.
		if (input_pointer != 4) {
			// Start the show.
			
			vdp_blocknmi();
			// Move the lines up in vdp RAM.
			vdp_getarea (input, 0, 3, 32, 3);
			vdp_putarea (input, 0, 0, 32, 3);
			
			vdp_getarea (input, 0, 6, 32, 3);
			vdp_putarea (input, 0, 3, 32, 3);
			
			vdp_getarea (input, 0, 9, 32, 3);
			vdp_putarea (input, 0, 6, 32, 3);
			
			// Get a good copy back.
			vdp_getarea (input, 0, 9, 32, 3);			
			vdp_releasenmi();
			
			// Send it back to ASCII.
			for(key_value = 0;key_value < 96;key_value++) {
				input[key_value] += ' ';
			}
					  		
			// We got a good copy again. Do something with it.
			speak(input_phoneme,input_phoneme_pointer);
			
			// Reset the input_pointer.
			input_pointer = 4;
			input_phoneme_pointer = 0;
			
			// Reposition input
		  	sprites[1].x = 4*8;
		  	sprites[1].y = 9*8;
			
			// Clear the input after use.
			for(key_value=4;key_value < 96;key_value++) {
				input[key_value] = ' ';
			}
		}
		
	} else {
		
		// Display
		for(cnt=0;cnt<2;cnt++) {
			
			key_value = kbphoneme1[key][cnt];			
			
			if (input_pointer < 96) {
				input[input_pointer] = key_value;
				input_pointer++;
				
				sprites[1].x += 8;
				
				if (sprites[1].x == 0) {
					sprites[1].y += 8;
				}
			}
		}
		
		// Compensate phoneme key
		if (key > 30) {
			key--;	
		}
		
		if (key > 10) {
			key--;
		}
		
		input_phoneme[input_phoneme_pointer++] = key;
	}
	
	// Update display.
	draw_string(0,9,input,96);
}

void ARPAbetSAY(void) {
	// Keyboard row and col.
	unsigned char row;
	unsigned char col;
	unsigned char done;
	
	vdp_blocknmi();
	// Set lite blue on blue.
	vdp_fillvram(0x2000,0x54,14*32*8);
	// All space.
	vdp_fillvram(0x1800,0x00,14*32);
	vdp_releasenmi();
	
	input_pointer = 4;
	input_phoneme_pointer = 0;
	
	clear_input();
	
	// draw_box();
	draw_string(0,9,"SAY:",4);
	input[0] = 'S';
	input[1] = 'A';
	input[2] = 'Y';
	input[3] = ':';
	
	setup_initialkb_charandsprite();
	
	vdp_enablescr();
	vdp_enablenmi();
	
	draw_string(1,1,"PRESS * TO EXIT",15);
	
	spr_enable = 1;
	
	row = 0;
	col = 0;
	done = 0;
	
	while(!done) {
		
		if (joypad_1 & PAD_UP) {
						
			// Now give it the correct column, xpos and key.
			if (row == 0) {
				
				row = 4;
				sprites[0].y += 64;
				sprites[0].x = 120;
				col = 0;
				
			} else if (row == 1) {
				
				row--;
				sprites[0].y -= 16;
				sprites[0].x -= 8;
				
			} else if (row == 2) {
				
				row--;
				sprites[0].y -= 16;
				sprites[0].x += 8;
								
			} else if (row == 3) {
				
				row--;
				sprites[0].y -= 16;
				sprites[0].x -= 8;
				
			} else {
				row--;
				sprites[0].y -= 16;
				sprites[0].x = 112;
				col = 4;
			}

		}
		
		if (joypad_1 & PAD_DOWN) {
						
			// Now give it the correct column, xpos and key.
			if (row == 0) {
				
				row++;
				sprites[0].y += 16;
				sprites[0].x += 8;
				
				if (col == 10) {
					col = 9;
					sprites[0].x -= 16;
				}
				
			} else if (row == 1) {
				
				row++;
				sprites[0].x -= 8;				
				sprites[0].y += 16;
				
			} else if (row == 2) {
				
				row++;
				sprites[0].y += 16;
				sprites[0].x += 8;
				
			} else if (row == 3) {
				row++;
				sprites[0].y += 16;
				sprites[0].x = 120;
				col = 0;				
			} else {
				row = 0;
				sprites[0].y -= 64;
				col = 5;				
			}
						
		}
		
		if (joypad_1 & PAD_RIGHT) {
			
			if (row == 0) {

				if (col == 10) {
					col = 0;
					sprites[0].x -= 160;
				} else {
					col++;
					sprites[0].x += 16;
				}
				
			} else if (row == 1) {

				if (col == 9) {
					col = 0;
					sprites[0].x -= 144;
				} else {
					col++;
					sprites[0].x += 16;
				}
				
			} else if (row == 2) {

				if (col == 9) {
					col = 0;
					sprites[0].x -= 144;
				} else {
					col++;
					sprites[0].x += 16;
				}
				
			} else if (row == 3) {
				if (col == 9) {
					col = 0;
					sprites[0].x -= 144;
				} else {
					col++;
					sprites[0].x += 16;
				}
				
			} else {
				// Must be row 3, do nothing only spacebar.
			}
			
		}
		
		if (joypad_1 & PAD_LEFT) {
			
			if (row == 0) {
				
				if (col == 0) {
					col = 10;
					sprites[0].x += 160;
				} else {
					col--;
					sprites[0].x -= 16;
				}
					
			} else if (row == 1) {

				if (col == 0) {
					col = 9;
					sprites[0].x += 144;
				} else {
					col--;
					sprites[0].x -= 16;
				}
				
			} else if (row == 2) {

				if (col == 0) {
					col = 9;
					sprites[0].x += 144;
				} else {
					col--;
					sprites[0].x -= 16;
				}
				
			} else if (row == 3) {

				if (col == 0) {
					col = 9;
					sprites[0].x += 144;
				} else {
					col--;
					sprites[0].x -= 16;
				}
			
			} else {
				// Must be row 4, do nothing only spacebar.
			}
						
		}
		
		// Are we pressing fire?
		if ( (joypad_1 & PAD_FIRE1) || (joypad_1 & PAD_FIRE2) ) {
			handle_key_phonemes(row,col);
		}
		
		if ( (keypad_1 == PAD_KEYSTAR) ) {
			done = 1;
			
			while(keypad_1 == PAD_KEYSTAR) {
				vdp_waitvblank(1);
			}
			
		}
		
		// When joystick still active, wait.
		while (joypad_1) vdp_waitvblank(1); // Wait Vblank while joy pressed
		
		// Wait at least a Vblank for the joystick to catch up.
		vdp_waitvblank(1);
	}
		
}

void checkjoy(void) {
	
	if (keypad_1 == PAD_KEY1) {
		
		if (speed < 20) {
			speed++;
		}
		
		while(keypad_1 == PAD_KEY1) {
			vdp_waitvblank(1);
		}			
	}
	
	if (keypad_1 == PAD_KEY2) {
		speed = 3;
		
		while(keypad_1 == PAD_KEY2) {
			vdp_waitvblank(1);
		}			

	}
	
	if (keypad_1 == PAD_KEY3) {
		
		if (speed > 3) {
			speed--;
		}
		
		while(keypad_1 == PAD_KEY3) {
			vdp_waitvblank(1);
		}			
	}

}

void ARPAbettest(void) {
	int cnt;
	
	unsigned char *phoneme;
	unsigned char *phonemes;
	unsigned char *word;
	unsigned char *phonemesample;

	setup_screen();
	printAt(1,1,"ARPABET SPEECH SYNTHESIZER TEST",0xF1);
	vdp_enablescr();
	vdp_enablenmi();
	
	// Run through all phonemes with an example.
	phonemes = (unsigned char *)HELLO;
	word = (unsigned char *)wHELLO;			
	for (cnt=0;cnt<39;cnt++) {
		
		printCentered(10,word);
		speak(phonemes,0);
		
		word += strlen(word) + 1;
		phonemes += *phonemes + 1;
		
		checkjoy();						
		vdp_waitvblank(50);
	}
	
	phoneme = (unsigned char *)wAA;
	phonemes = (unsigned char *)BALM;
	word = (unsigned char *)wBALM;
	phonemesample = (unsigned char *)ph_AA_2bitDPCM;
	
	for(cnt=0;cnt<NUM_PHONEMES;cnt++) {
		// print the phoneme
		printCentered(8,phoneme);
		printCentered(9,"AS IN");
		// print the word
		printCentered(10,word);
		
		play_sample(phonemesample,phoneme_lengths[cnt]);
		vdp_waitvblank(30);
		speak(ASIN,0);
		vdp_waitvblank(30);
		speak(phonemes,0);
		
		phoneme += strlen(phoneme) + 1;
		word += strlen(word) + 1;
		phonemes += *phonemes + 1;
		phonemesample += phoneme_lengths[cnt];
		
		checkjoy();		
		vdp_waitvblank(50);
	}

	// Clear lines.
	clear_line(8);
	clear_line(9);
	
	// print the word
	printCentered(10,wTHANKYOU);
	speak(THANKYOU,0);
	vdp_waitvblank(50);	
}

/*
 * Main menu.
 *
 * ARPABET SAY
 * ARPABET TEST
 * INSULT ME
 *
 */
#define NUM_MENU_ITEMS 3
const char menu_table[NUM_MENU_ITEMS][21] = {
	"    ARPABET SAY     ",
	"    ARPABET TEST    ",
	"     INSULT ME      "
};

unsigned char main_menu(void) {
	unsigned char result;
	unsigned char done;
	unsigned char cnt;
	
	setup_screen();
	
	printAt(2,2,"ARPABET FOR THE COLECOVISION",0xF1);
	printAt(1,4,"SPEECH SYNTHESIS USING PHONEMES",0xF1);
	printAt(2,6,"USE JOYSTICK FOR MENU BELOW",0xF1);
	printAt(2,8,"PRESS FIRE TO SELECT OPTION",0xF1);
	
	vdp_enablescr();
	vdp_enablenmi();
	
	/* default action */
	result = 0;

	done = 0;
		
	/* Setup our menu */
	for (cnt=0;cnt<NUM_MENU_ITEMS;cnt++) {
		printAt(6,12+cnt,menu_table[cnt],0xF1);	
	}
	
	color_offset = 0;
	
	while(!done) {
		
		/* wait for vblank */
		vdp_waitvblank(1);
		
		/* Color selected line */
		color8x8at(6,12+result,20,0xF0+colors[color_offset]);
		
		/* Check joystick */
		if (joypad_1 & PAD_UP) {
			
			if (result != 0) {
				color8x8at(6,12+result,20,0xF1);
				result--;
				color8x8at(6,12+result,20,0xF0+colors[color_offset]);
			} else {
				color8x8at(6,12+result,20,0xF1);
				result = NUM_MENU_ITEMS-1;
				color8x8at(6,12+result,20,0xF0+colors[color_offset]);
			}
			
			while (joypad_1 & PAD_UP) {
				;
			}
			
		} else if (joypad_1 & PAD_DOWN) {
			
			if ( result != (NUM_MENU_ITEMS - 1) ) {
				color8x8at(6,12+result,20,0xF1);
				result++;
				color8x8at(6,12+result,20,0xF0+colors[color_offset]);
			} else {
				color8x8at(6,12+result,20,0xF1);
				result=0;
				color8x8at(6,12+result,20,0xF0+colors[color_offset]);
			}
			
			while(joypad_1 & PAD_DOWN) {
				;	
			}
		}
		
		if ( (joypad_1 & PAD_FIRE1) || (joypad_1 & PAD_FIRE2) ) {
			
			// Wait for release.
			while(joypad_1) vdp_waitvblank(1);
			
			done = 1;
		}
		
		color_offset++;
		
		if (color_offset == 56) {
			color_offset = 0;
		}	
	}
	
	return(result);
}

//---------------------------------------------------------------------------------
void main (void) {
	unsigned char result;
	
	// Overall sample speed.
	speed = 3;
	
	show_intro();

	while(1) {
		result = main_menu();
		
		switch (result) {
			
			case 0:
				ARPAbetSAY();
			break;
			
			case 1:
				// Jump to say_mode
				ARPAbettest();
			break;
			
			case 2:
				insult_me();
			break;
									
			default:
			break;
			
		}	
		
	}
	// startup module will issue a soft reboot if it comes here 
}

